% $Id: srcsine.m,v 1.6 1998/07/20 23:24:53 ericb Exp $
% Copyright (C) 1998, Hewlett-Packard Company, all rights reserved
% Written by Fred Christensen

% Example of source SINE mode operation.

clear all;

%**************************************************************************
% Hardware Setup:
%    1. Set the Logical Address of all VXI cards.
%    2. Modify the hpe1432('init' ...) command in section #1 below to
%       include the logical addresses of those cards you wish to use.
%    3. Connect first source channel in SOURCE_CHS list below to first 
%       input channel in INPUT_CHS list below.
%             OR
%       Look at any source channel in SOURCE_CHS list below with a scope.
%**************************************************************************

%**************************************************************************
% A. Defines
%**************************************************************************

% A1. Defines that can be changed by the user ...

	% list of input channels to turn on...
	%    - set this to [] if you do not have or want to turn on any input channels
   %	  - this can be a sparse array (such as [1 3 5])
  
INPUT_CHS  = [1];
	% list of source channels to turn on...
   % 	  - first source channel starts at 4097
   %	  - this can be a sparse array (such as [4097 4099])
SOURCE_CHS = [4097];
   
CLOCK_FREQ = 51200.;		% Module clock frequency
SPAN = 1000.;				% Module span (used by inputs & for trigger timing)
BLOCKSIZE = 1024;			% Input Blocksize
BLOCKSIZE_S = 1024;		% Source Blocksize

RANGE_I = 1.0;				% Input range

SPAN_S = 1000.;			% Source span
AMP_S = 1.0;				% Source Amplitude (= Source Range * Amplitude Scale)
RAMP_TIME = 0.0;		   % Source ramp time in seconds
SINE_FREQ = 160.0;		% Source sine freq (for sine modes only)

%******************************************************************************

% A2. Other Defines

SOURCE_MODE = 'SOURCE_MODE_SINE';	% See HPE1432 Help for possible modes
% Mapping arrays... use these to make user friendly disp statements.
% You can get the mappings from the HPE1432A Help documentation (look
% under the help for the associated function)
SOURCE_MODES = {'SOURCE_MODE_SINE' 'SOURCE_MODE_BSINE' 'SOURCE_MODE_RAND' ...
      'SOURCE_MODE_BRAND' 'SOURCE_MODE_RANDZ' 'SOURCE_MODE_BRANDZ' ...
      'SOURCE_MODE_ARB' 'SOURCE_MODE_BARB'};

% Arrays for plots later on during the measurement
Xtime = 0:1/(SPAN*2.56):(BLOCKSIZE-1)/(SPAN*2.56);  % time axis values
Xfreq = 0:SPAN/(BLOCKSIZE/2.56):SPAN;   % Freq axis values

%****************************************************************************
% 1. Initialize Software connection to Hardware and Setup Channel Groups
%****************************************************************************

disp('1. Initializing Hardware...');

% Change the logical address numbers below to match your hardware setup
[status,session] = hpe1432('init','VXI0::8::INSTR',1,1);
if double(status) < 0
   disp('NOTE: Check to make sure that the logical addresses in the hpe1432');
   disp('      "init" statement match those of your hardware.  Also make sure');
   disp('      that the Resource Manager was run successfully.');
   disp(' ');
end
hpe1432_check_status(session, status); % Make sure to always check for errors in
													% the status return value.


% Delete any leftover channel groups
[status] = hpe1432('deleteAllChanGroups',session);
hpe1432_check_status(session, status);

% Find out about channels present and create appropriate groups of channels

[status,TotChans,NumInpChans,NumSrcChans,NumTachChans] = ...
		hpe1432('getNumChans',session);
hpe1432_check_status(session, status);

TotChans=double(TotChans);
NumInpChans=double(NumInpChans);
NumSrcChans=double(NumSrcChans);
NumTachChans=double(NumTachChans);

inp_chs_avail = 1:NumInpChans;						% Create list of input channel ids
source_chs_avail = 4097:4097+(NumSrcChans-1);	% Create list of source channel ids
tot_chs_avail = [inp_chs_avail source_chs_avail];	% Create list of all channel ids

	% Check to make sure that all channels in SOURCE_CHS and INPUT_CHS are 
	% actually present in the system
for i=1:length(SOURCE_CHS)
   if isempty(find(source_chs_avail == SOURCE_CHS(i)))
      disp('Error: SOURCE_CHS contains a channel id which is not present')
   end
end
disp(['	Source Channels On: ' num2str(SOURCE_CHS)]);

for i=1:length(INPUT_CHS)
   if isempty(find(inp_chs_avail == INPUT_CHS(i)))
      disp('Error: INPUT_CHS contains a channel id which is not present');
      disp('       Set INPUT_CHS = [] if there are no input channels');
   end
end
disp(['	Input Channels On: ' num2str(INPUT_CHS)]);

	%create a group of all physical channels and then turn them off in case some
	%previous activity left them on
[status,global_gid] = hpe1432('createChannelGroup',session, ...
		length(tot_chs_avail),tot_chs_avail);
hpe1432_check_status(session, status);
[status] = hpe1432('setActive',session,global_gid,'CHANNEL_OFF');
hpe1432_check_status(session, status);

   %create a group of just the inputs that will be active
if ~isempty(INPUT_CHS)   
   [status,gid_i] = hpe1432('createChannelGroup',session,length(INPUT_CHS),INPUT_CHS);
   hpe1432_check_status(session, status);
end   
   %Create a group of just the sources that will be active
[status,gid_s] = hpe1432('createChannelGroup',session,length(SOURCE_CHS),SOURCE_CHS);
hpe1432_check_status(session, status);

   %Create a global group of inputs and sources
if isempty(INPUT_CHS)
   [status,gid_g] = hpe1432('createChannelGroup',session,...
      length([SOURCE_CHS]),[SOURCE_CHS]);
   hpe1432_check_status(session, status);
else   
   [status,gid_g] = hpe1432('createChannelGroup',session,...
      length([INPUT_CHS SOURCE_CHS]),[INPUT_CHS SOURCE_CHS]);
   hpe1432_check_status(session, status);
end

%**********************************************************************
% 2. Setup "Global" parameters
%**********************************************************************

disp('2. Setup "Global" parameters...');

[status] = hpe1432('setClockFreq',session, gid_g, CLOCK_FREQ);
hpe1432_check_status(session, status);
[status, gotClockFreq] = hpe1432('getClockFreq', session, gid_g);
hpe1432_check_status(session, status);
disp(['	Clock Freq: ' num2str(gotClockFreq)]);

% Note: Span should be set even on E1434 modules so that the source trigger
%       can line up with decimated input data (when other input modules are present).
[status] = hpe1432('setSpan',session,gid_g,SPAN);
hpe1432_check_status(session, status);
[status, gotSpan] = hpe1432('getSpan',session,gid_g);
hpe1432_check_status(session, status);
disp(['	Span: ' num2str(gotSpan)]);

%*********************************************************************
% 3. Setup Input Channel parameters (when input modules are present)
%*********************************************************************

% Only set these up if input channels were specified (so this example
% can also be used when no input modules are present)
if INPUT_CHS
   disp('3. Setup Input Channel parameters...');
   
   [status] = hpe1432('setActive',session,gid_i,'CHANNEL_ON');
	hpe1432_check_status(session, status);
   
   [status] = hpe1432('setBlocksize',session,gid_i,BLOCKSIZE);
	hpe1432_check_status(session, status);
   [status, gotBlocksize] = hpe1432('getBlocksize',session,gid_i);
   hpe1432_check_status(session, status);
   disp(['	Input Blocksize: ' num2str(double(gotBlocksize))]);
   
   [status] = hpe1432('setRange',session,gid_i,RANGE_I);
	hpe1432_check_status(session, status);
   [status, gotRange] = hpe1432('getRange',session,gid_i);
   hpe1432_check_status(session, status);
   disp(['	Input Range: ' num2str(gotRange)]);
   
   % Input span already set above in global parms (so just get it for
   % info purposes)
   [status,gotSpan] = hpe1432('getSpan',session,gid_i);
   hpe1432_check_status(session, status);
   disp(['	Input span = ' num2str(gotSpan)]);
  
   [status] = hpe1432('setDataMode',session,gid_i,'BLOCK_MODE');
   hpe1432_check_status(session, status);
   
end

%*********************************************************************
% 4. Setup Source Channel parameters
%*********************************************************************

disp('4. Setup Source Channel parameters...');

% 4.1 Get and display source board info

% Get some source info and display it... this can be useful when reporting
% bugs or checking to make sure that all source boards have the same firmware
% revs in ROM.  Note: Sources must be turned on and the source buffer properly
% initialized for this function to work.  Also srcGetRev should not be called
% during an Arb measurement since it uses the source transfer buffers.
[status] = hpe1432('setActive',session,gid_s,'CHANNEL_ON');
hpe1432_check_status(session, status);

[status] = hpe1432('setSrcBufferInit',session,gid_s,'SRC_BUFFER_INIT_XFER');
hpe1432_check_status(session, status);
for i=1:length(SOURCE_CHS),
   [status, romid, romdate, bdid, bddate] = hpe1432('srcGetRev', session, ...
      SOURCE_CHS(i));
   hpe1432_check_status(session, status);
   disp(['	SOURCE CHAN: ' num2str(SOURCE_CHS(i)) ...
         ' romid=' num2str(dec2hex(double(romid(1)))) ...
         ' romdate=' num2str(dec2hex(double(romdate(1)))) ...
         ' bdid=' num2str(dec2hex(double(bdid(1)))) ...
         ' bddate=' bddate]);
end

% 4.2 Generic source parameters
[status] = hpe1432('setActive',session,gid_s,'CHANNEL_ON');
hpe1432_check_status(session, status);
   
[status] = hpe1432('setSourceBlocksize',session, gid_s,BLOCKSIZE_S);
hpe1432_check_status(session, status);
[status, gotSourceBlocksize] = hpe1432('getSourceBlocksize', session, gid_s);
hpe1432_check_status(session, status);
disp(['	Source Blocksize: ' num2str(double(gotSourceBlocksize))]);
   
[status] = hpe1432('setRampRate',session,gid_s,RAMP_TIME);
hpe1432_check_status(session, status);
[status, gotRampRate] = hpe1432('getRampRate', session, gid_s);
hpe1432_check_status(session, status);
disp(['	Source Ramp Rate: ' num2str(gotRampRate)]);
   
    % Source uses a log DAC so pick the closest Range above AMP_S and
    % then use AmpScale to scale the output down to the desired amplitude.
    % (Output voltage = Range * AmpScale)    
[status] = hpe1432('setRange',session,gid_s,AMP_S*10^(.375/20));
hpe1432_check_status(session, status);
[status, gotRange] = hpe1432('getRange',session,gid_s);
hpe1432_check_status(session, status);
disp(['	Source Range: ' num2str(gotRange)]);
      
[status] = hpe1432('setAmpScale',session, gid_s, AMP_S/double(gotRange));
hpe1432_check_status(session, status);
[status, gotAmpScale] = hpe1432('getAmpScale',session,gid_s);
hpe1432_check_status(session, status);
disp(['	Source Amp Scale: ' num2str(gotAmpScale)]);

% 4.3 Source mode specific parameters 
[status] = hpe1432('setSourceMode',session,gid_s,SOURCE_MODE);
hpe1432_check_status(session, status);
[status, gotSourceMode] = hpe1432('getSourceMode', session, gid_s);
hpe1432_check_status(session, status);
disp(['	Source Mode: ' SOURCE_MODES{double(gotSourceMode)-169}]);
   
[status] = hpe1432('setSourceSpan', session, gid_s, SPAN_S);        
hpe1432_check_status(session, status);
[status, gotSourceSpan] = hpe1432('getSourceSpan', session, gid_s);
hpe1432_check_status(session, status);
disp(['	Source Span: ' num2str(gotSourceSpan) ' Hz']);

% SINE mode parameters
[status] = hpe1432('setSineFreq',session, gid_s,SINE_FREQ);
hpe1432_check_status(session, status);
[status, gotSineFreq] = hpe1432('getSineFreq', session, gid_s);
hpe1432_check_status(session, status);
disp(['	Source Sine Freq: ' num2str(gotSineFreq)]);

[status] = hpe1432('setSinePhase',session, gid_s,0);     
hpe1432_check_status(session, status);
	
%*********************************************************************
% 5. Setup Triggering
%*********************************************************************
disp('5. Setup Triggering...');

% Let all channels auto arm themselves
[status] = hpe1432('setArmMode',session,gid_g,'AUTO_ARM');
hpe1432_check_status(session, status);      

% Let all channels auto trigger themselves
[status] = hpe1432('setAutoTrigger',session,gid_g,'AUTO_TRIGGER');
hpe1432_check_status(session, status);
   
% No trigger channels on since auto triggering is being used
[status] = hpe1432('setTriggerChannel',session,gid_g,'CHANNEL_OFF');
hpe1432_check_status(session, status);

%****************************************************************
% 6. Pre-Load Source Arb Data buffers
%****************************************************************

%****************************************************************
% 7. Start The Measurement
%****************************************************************
disp('7. Start Measurement...');

% Do an Auto Zero just before starting the measurement... shouldn't need to
% do an autozero during a measurement even when changing ranges.
[status] = hpe1432('autoZero',session,gid_g);
hpe1432_check_status(session, status);

% Start measurement
disp('Measurement running... Select Stop under Measurement menu to stop it.');
[status] = hpe1432('initMeasure',session,gid_g);
hpe1432_check_status(session, status);

scan=1;
hold off;
stop = 0;

% Infinite loop
while ~stop,
   if scan==1
      % Allow user to exit the while loop from Figure menu
      clf;
      hMeasMnu = uimenu(gcf);
      set(hMeasMnu,'Label','&Measurement');
      hMeasStopMnu = uimenu(hMeasMnu);
      set(hMeasStopMnu,'Label','Stop');
      set(hMeasStopMnu,'Callback','disp(''Measurement Stopped''),stop=1;');
      zoom on	% enable zoom for the user
   end

   % If we have some input channels on then read the data, fft it, and plot it   
   if ~isempty(INPUT_CHS)   
      [status,data,count] = hpe1432('readFloat64Data',session,gid_i,'TIME_DATA', ...
         BLOCKSIZE*length(INPUT_CHS),'WAIT_FLAG');
      hpe1432_check_status(session, status);
      
      data_mat = reshape(data,BLOCKSIZE,length(INPUT_CHS));
      data_fft = (2/BLOCKSIZE)*fft(data_mat(:,1)./RANGE_I);            %  fft channel 1
  	   data_fft = (data_fft).*conj(data_fft);   	% get magnitude  
     	data_fft= data_fft(1:floor(BLOCKSIZE/2.56)+1,:);   % alias free frequency range
     	data_fft(find(data_fft == 0)) = 1E-20;	% Eliminate any zero entries before doing log
     	data_fft = 10*log10(data_fft);		% Convert to dB
        
      if scan == 1	% Setup plots the first time through
         % Do all this stuff instead of just using the "plot" command
         % for speed issues (redrawing will be lots faster)
      
         % Time Plot
         haxestime = axes('Position',[.1 .625 .85 .3]);		% Axes for plotting time   
         h1=plot(Xtime,data_mat(:,1));drawnow;   % plot time data for channel 1
         title('Time');
         xlabel('Time (sec)');
         ylabel('Amplitude (V)');
      	set(h1,'EraseMode','background');
      	h1axis=get(h1,'parent');
      	set(h1axis,'YLimMode','manual');
         set(h1axis,'ylim',[-RANGE_I RANGE_I]);
         
         % Freq Plot
      	haxesfreq = axes('Position',[.1 .125 .85 .3]);
         h2=plot(Xfreq,data_fft(:,1));
         title('Spectrum');
         xlabel('Frequency (Hz)');
         ylabel('dBfs'), drawnow;  % plot freq data channel 1
      	set(h2,'EraseMode','background');
      	h2axis=get(h2,'parent');
      	set(h2axis,'YLimMode','manual');
         set(h2axis,'ylim',[-150 0]);
            % get rid of exponent notation in X labels (X tick labels get
            % messed up for zoom though when this is done)
         %set(haxesfreq,'XTickLabel',sprintf('%4.0f|',get(haxesfreq,'XTick')));
         
      else
         set(gcf,'CurrentAxes',haxestime);
         set(h1,'YData',data_mat(:,1)),drawnow;
         set(gcf,'CurrentAxes',haxesfreq);
         set(h2,'YData',data_fft(:,1)),drawnow;
      end
   else
      pause(1);  %Give time for user to press "stop" in Figure window
   end   % end isempty

scan=scan+1;    % loop counter
end

% Code never gets here because of infinite loop (Need to add user breakout
% from this inifinite loop)
for chan = 1:length(SOURCE_CHS),
   
   % It is a good thing to check several source functions after running a
   % measurement just to make sure things went well.  
   
   % Check if the source DSP somehow fell out of real-time (shouldn't happen)
   [status, srcUnderrun] = hpe1432('checkSrcUnderrun', session, SOURCE_CHS(chan));
   hpe1432_check_status(session, status);
   % Check if something overloaded the source outputs.
   [status, srcOverload] = hpe1432('checkSrcOverload', session, SOURCE_CHS(chan));
   hpe1432_check_status(session, status);
   % Check if the sources might have shut themselves down (such as an emergency
   %   shutdown -- commanded by the user or as a result of an overload, or from
   %   the shutdown pin)
   [status, srcShutdown] = hpe1432('checkSrcShutdown', session, SOURCE_CHS(chan));
   hpe1432_check_status(session, status);
   if double(srcUnderrun) | double(srcOverload) | ...
         double(srcShutdown)
      disp(['Chan ' num2str(SOURCE_CHS(chan)) '::  '...
         'srcOverread: ' num2str(double(srcOverread)) '  srcUnderrun: ' ...
         num2str(double(srcUnderrun)) '  srcOverload: ' num2str(double(srcOverload)) ...
         '  srcShutdown: ' num2str(double(srcShutdown))]);
   end
end

[status] = hpe1432('close',session);
hpe1432_check_status(session, status);


